<?php

namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use App\Models\KehadiranModel;
use App\Models\SiswaModel;
use App\Models\TahunAjaranModel;
use App\Models\KelasModel;

class Kehadiran extends BaseController
{
    protected $kehadiranModel;
    protected $siswaModel;
    protected $tahunAjaranModel;
    protected $kelasModel;

    public function __construct()
    {
        $this->kehadiranModel = new KehadiranModel();
        $this->siswaModel = new SiswaModel();
        $this->tahunAjaranModel = new TahunAjaranModel();
        $this->kelasModel = new KelasModel();
    }
    
    // Fungsi bantu untuk mendapatkan ID kelas yang diajar oleh guru yang sedang login (sebagai wali kelas)
    private function getGuruKelasIds()
    {
        $user_id = session()->get('user_id');
        $kelas_guru = $this->kelasModel->where('guru_kelas_id', $user_id)->findAll();
        
        $kelas_ids = [];
        foreach ($kelas_guru as $kelas) {
            $kelas_ids[] = $kelas['id_kelas'];
        }
        return $kelas_ids;
    }
    
    public function index()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (!$is_guru_kelas) {
            $data = [
                'title' => 'Input Kehadiran',
                'siswa' => [],
                'data_kehadiran' => [],
                'message' => 'Anda tidak terdaftar sebagai guru kelas. Halaman ini hanya bisa diakses oleh wali kelas.'
            ];
            return view('guru/kehadiran/index', $data);
        }

        $tahun_ajaran = $this->tahunAjaranModel->where('aktif', 1)->first();
        
        if (empty($tahun_ajaran)) {
            $data = [
                'title' => 'Input Kehadiran',
                'siswa' => [],
                'data_kehadiran' => [],
                'message' => 'Tidak ada tahun ajaran atau semester aktif yang ditemukan.'
            ];
            return view('guru/kehadiran/index', $data);
        }

        $siswa = $this->siswaModel
            ->select('siswa.id, siswa.nis, siswa.nama_lengkap, kelas.nama_kelas')
            ->join('kelas', 'kelas.id_kelas = siswa.kelas_id')
            ->whereIn('siswa.kelas_id', $guru_kelas_ids)
            ->findAll();

        $data_kehadiran = [];
        $kehadiran_db = $this->kehadiranModel
            ->where('tahun_ajaran_id', $tahun_ajaran['id'])
            ->where('semester', $tahun_ajaran['semester'])
            ->findAll();

        foreach ($kehadiran_db as $item) {
            $data_kehadiran[$item['siswa_id']] = $item;
        }

        $data = [
            'title' => 'Input Kehadiran',
            'siswa' => $siswa,
            'data_kehadiran' => $data_kehadiran
        ];

        return view('guru/kehadiran/index', $data);
    }
    
    public function save()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (!$is_guru_kelas) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
            return redirect()->to(base_url('guru/kehadiran'));
        }

        $sakit_data = $this->request->getPost('sakit');
        $izin_data = $this->request->getPost('izin');
        $tanpa_keterangan_data = $this->request->getPost('tanpa_keterangan');

        $tahun_ajaran = $this->tahunAjaranModel->where('aktif', 1)->first();
        if (empty($tahun_ajaran)) {
            session()->setFlashdata('error', 'Tidak ada tahun ajaran atau semester aktif yang ditemukan.');
            return redirect()->to(base_url('guru/kehadiran'));
        }

        if (is_array($sakit_data)) {
            foreach ($sakit_data as $siswa_id => $sakit) {
                $siswa_info = $this->siswaModel->find($siswa_id);
                if (!in_array($siswa_info['kelas_id'], $guru_kelas_ids)) {
                    session()->setFlashdata('error', 'Anda tidak memiliki izin untuk menyimpan data kehadiran siswa di kelas tersebut.');
                    return redirect()->back();
                }

                $data = [
                    'siswa_id' => $siswa_id,
                    'semester' => $tahun_ajaran['semester'],
                    'tahun_ajaran_id' => $tahun_ajaran['id'],
                    'sakit' => $sakit,
                    'izin' => isset($izin_data[$siswa_id]) ? $izin_data[$siswa_id] : 0,
                    'tanpa_keterangan' => isset($tanpa_keterangan_data[$siswa_id]) ? $tanpa_keterangan_data[$siswa_id] : 0,
                ];

                $existing_data = $this->kehadiranModel
                    ->where('siswa_id', $siswa_id)
                    ->where('tahun_ajaran_id', $tahun_ajaran['id'])
                    ->first();

                if ($existing_data) {
                    $this->kehadiranModel->update($existing_data['id'], $data);
                } else {
                    $this->kehadiranModel->insert($data);
                }
            }
        }
        session()->setFlashdata('success', 'Rekap kehadiran berhasil disimpan.');
        return redirect()->to(base_url('guru/kehadiran'));
    }
}