<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\UserModel;
use App\Models\TahunAjaranModel;
use App\Models\KelasModel; // Penting: Import KelasModel

class Kelas extends BaseController
{
    protected $kelasModel; // Deklarasikan properti untuk menyimpan instance model
    protected $userModel;
    protected $tahunAjaranModel;

    public function __construct()
    {
        // Inisialisasi model di konstruktor agar bisa digunakan di seluruh metode
        $this->kelasModel = new KelasModel();
        $this->userModel = new UserModel();
        $this->tahunAjaranModel = new TahunAjaranModel();
    }

    public function index()
    {
        // Ambil semua data kelas dari database menggunakan model
        $daftarKelas = $this->kelasModel->findAll();

        $data = [
            'title'     => 'Manajemen Kelas', // Judul halaman untuk layout
            'daftarKelas' => $daftarKelas     // Data kelas yang akan dilewatkan ke view
        ];

        // Memuat view 'admin/kelas/index' dan melewatkan variabel $data
        return view('admin/kelas/index', $data);
    }

    public function create() 
	{
        // AMBIL DATA GURU
        $gurus = $this->userModel
            ->select('id, nama')
            ->orderBy('nama', 'ASC')
            ->findAll();
        
        $tahunAjaranList = $this->tahunAjaranModel
        ->select('id, nama_tahun')
        ->orderBy('nama_tahun', 'DESC')
        ->findAll();
        
        $data = [
            'title' => 'Tambah Kelas Baru', // Judul halaman untuk layout
            'validation' => \Config\Services::validation(), // Untuk menampilkan error validasi
            'gurus' => $gurus, // KIRIM DATA GURU KE VIEW
            'tahunAjaranList' => $tahunAjaranList,
        ];

        // Memuat view form penambahan kelas
        return view('admin/kelas/create', $data);
    }
    
	public function store() 
	{
        // Lakukan validasi input
        if (!$this->validate([
            'nama_kelas' => [
                'rules' => 'required|min_length[1]|is_unique[kelas.nama_kelas,tahun_ajaran_id,{tahun_ajaran_id}]', // Tambahkan validasi unik gabungan
                'errors' => [
                    'required' => 'Nama Kelas harus diisi.',
                    'min_length' => 'Nama Kelas minimal 1 karakter.',
                    'is_unique' => 'Nama Kelas ini sudah ada untuk tahun ajaran yang dipilih.'
                ]
            ],
            'guru_kelas_id' => [
                'rules' => 'permit_empty|is_natural_no_zero',
                'errors' => [
                    'is_natural_no_zero' => 'ID Guru Kelas harus angka positif.'
                ]
            ],
            'tahun_ajaran_id' => [
                'rules' => 'required|is_natural_no_zero',
                'errors' => [
                    'required' => 'Tahun Ajaran harus diisi.',
                    'is_natural_no_zero' => 'ID Tahun Ajaran harus angka positif.'
                ]
            ],
        ])) {
            // Jika validasi gagal, kembali ke form create dengan error dan input lama
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        // Simpan data ke database
        $this->kelasModel->save([
            'nama_kelas'      => $this->request->getVar('nama_kelas'),
            'guru_kelas_id'   => $this->request->getVar('guru_kelas_id'),
            'tahun_ajaran_id' => $this->request->getVar('tahun_ajaran_id'),
        ]);

        // Set flashdata untuk pesan sukses
        session()->setFlashdata('success', 'Data kelas berhasil ditambahkan.');

        // Redirect kembali ke halaman manajemen kelas
        return redirect()->to(base_url('admin/kelas'));
    }
    
	public function edit($id_kelas) // Pastikan nama parameter sesuai dengan route (id_kelas)
    {
        // Cari data kelas berdasarkan ID
        $kelas = $this->kelasModel->find($id_kelas);

        // Jika kelas tidak ditemukan, redirect ke halaman daftar kelas dengan pesan error
        if (empty($kelas)) {
            session()->setFlashdata('error', 'Data kelas tidak ditemukan.');
            return redirect()->to(base_url('admin/kelas'));
        }

        // AMBIL DATA GURU
        $gurus = $this->userModel
            ->select('id, nama')
            ->orderBy('nama', 'ASC')
            ->findAll();
        
        $tahunAjaranList = $this->tahunAjaranModel
            ->select('id, nama_tahun')
            ->orderBy('nama_tahun', 'DESC')
            ->findAll();
        
        $data = [
            'title' => 'Edit Kelas', // Judul halaman
            'validation' => \Config\Services::validation(), // Untuk validasi
            'gurus'      => $gurus,
            'tahunAjaranList' => $tahunAjaranList,
            'kelas' => $kelas // Data kelas yang akan diisi ke form
        ];

        // Memuat view form edit kelas
        return view('admin/kelas/edit', $data);
    }

    // --- FUNGSI BARU: MEMPERBARUI DATA KELAS ---
    public function update($id_kelas) // Pastikan nama parameter sesuai dengan route
    {
        // Aturan validasi. Perhatikan aturan is_unique untuk nama_kelas
        // Kita perlu mengecualikan data kelas yang sedang diedit dari pemeriksaan unik
        $namaKelasRule = 'required|min_length[1]|is_unique[kelas.nama_kelas,id_kelas,' . $id_kelas . ',tahun_ajaran_id,{tahun_ajaran_id}]';

        if (!$this->validate([
            'nama_kelas' => [
                'rules' => $namaKelasRule,
                'errors' => [
                    'required' => 'Nama Kelas harus diisi.',
                    'min_length' => 'Nama Kelas minimal 1 karakter.',
                    'is_unique' => 'Nama Kelas ini sudah ada untuk tahun ajaran yang dipilih.'
                ]
            ],
            'guru_kelas_id' => [
                'rules' => 'permit_empty|is_natural_no_zero',
                'errors' => [
                    'is_natural_no_zero' => 'ID Guru Kelas harus angka positif.'
                ]
            ],
            'tahun_ajaran_id' => [
                'rules' => 'required|is_natural_no_zero',
                'errors' => [
                    'required' => 'Tahun Ajaran harus diisi.',
                    'is_natural_no_zero' => 'ID Tahun Ajaran harus angka positif.'
                ]
            ],
        ])) {
            // Jika validasi gagal, kembali ke form edit dengan error dan input lama
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        // Perbarui data di database
        $this->kelasModel->update($id_kelas, [
            'nama_kelas'      => $this->request->getVar('nama_kelas'),
            'guru_kelas_id'   => $this->request->getVar('guru_kelas_id'),
            'tahun_ajaran_id' => $this->request->getVar('tahun_ajaran_id'),
        ]);

        session()->setFlashdata('success', 'Data kelas berhasil diperbarui.');
        return redirect()->to(base_url('admin/kelas'));
    }

    public function delete($id_kelas) 
	{
        // Hapus data dari database
        $this->kelasModel->delete($id_kelas);

        session()->setFlashdata('success', 'Data kelas berhasil dihapus.');
        return redirect()->to(base_url('admin/kelas'));
    } // Ini akan kita isi setelah edit/update
}