<?php namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use App\Models\SiswaModel; 
use App\Models\MataPelajaranModel; 
use App\Models\KelasModel;
use App\Models\SekolahModel;
use App\Models\TahunAjaranModel; 
use App\Models\NilaiModel; // Nilai Komponen
use App\Models\NilaiRaporModel; // Nilai Akhir/Rapor

class DashboardGuru extends BaseController
{
    // KOREKSI FINAL: Jumlah komponen yang diinput (2 NH + 1 PTS + 1 PAS)
    const MAX_KOMPONEN_PER_SISWA = 4; 

    protected $siswaModel;
    protected $mapelModel;
    protected $kelasModel;
	protected $sekolahModel;
    protected $tahunAjaranModel;
    protected $nilaiModel;
    protected $nilaiRaporModel;

    public function __construct()
    {
        // ASUMSI: Model-model ini sudah Anda buat dan siap digunakan
        $this->siswaModel = new SiswaModel();
        $this->mapelModel = new MataPelajaranModel();
        $this->kelasModel = new KelasModel();
		$this->sekolahModel = new SekolahModel();
        $this->tahunAjaranModel = new TahunAjaranModel();
        $this->nilaiModel = new NilaiModel(); // Tabel 'nilai'
        $this->nilaiRaporModel = new NilaiRaporModel(); // Tabel 'nilai_rapor'
    }

    public function index()
    {
        // KOREKSI PENTING: Mengambil ID Guru dari sesi 'user_id'
        $guruId = session()->get('user_id'); 
        
        if (!$guruId) {
             $data = ['title' => 'Dashboard Guru', 'progressInputNilai' => [], 'kelas' => null, 'chartData' => ['total' => 0, 'selesai' => 0, 'belum_selesai' => 0]];
             return view('guru/dashboard', $data);
        }
		
		helper('tanggal');
		// TAMBAHKAN BARIS INI: Ambil data sekolah (biasanya baris pertama/satu-satunya)
		$dataSekolah = $this->sekolahModel->first();
        
		// 1. Ambil data Kelas yang diajar wali kelas
        // Asumsi: kolom ID kelas adalah 'id_kelas'
        $kelasWali = $this->kelasModel->where('guru_kelas_id', $guruId)->first();
        
        // 2. Tentukan Tahun Ajaran dan Semester Aktif
        // Menggunakan kolom 'aktif' yang sudah dikonfirmasi
        $tahunAjaranAktif = $this->tahunAjaranModel->where('aktif', 1)->first(); 

        if (!$kelasWali || !$tahunAjaranAktif) {
             $data = ['title' => 'Dashboard Guru', 'progressInputNilai' => [], 'kelas' => null];
             // Pastikan variabel ada di data, meskipun isinya null
             $data['kelas'] = $kelasWali; 
             $data['tahunAjaran'] = $tahunAjaranAktif;
             // Jika data dasar tidak ada, ringkasan data juga 0
             $data['chartData'] = ['total' => 0, 'selesai' => 0, 'belum_selesai' => 0];
             return view('guru/dashboard', $data);
        }

        $kelasId = $kelasWali['id_kelas'];
        $tahunAjaranId = $tahunAjaranAktif['id'];
        $semester = $tahunAjaranAktif['semester'];

        // 3. Hitung Progres Input Nilai
        $progressInputNilai = $this->getProgressInputNilai(
            $kelasId, 
            $tahunAjaranId, 
            $semester
        );
		
		// --- NEW: Hitung Ringkasan Data untuk Chart ---
        $countSelesai = 0;
        foreach ($progressInputNilai as $item) {
            // Kita hitung selesai jika persentase 100
            if ($item['persentase'] == 100) {
                $countSelesai++;
            }
        }
        $totalMapel = count($progressInputNilai);
        $countBelumSelesai = $totalMapel - $countSelesai;
        
        $chartData = [
            'total' => $totalMapel,
            'selesai' => $countSelesai,
            'belum_selesai' => $countBelumSelesai,
        ];

        $data = [
            'title' => 'Dashboard Wali Kelas',
            'user'  => session()->get('user'),
            'kelas' => $kelasWali,
            'sekolah' => $dataSekolah, // Ini yang akan dibaca oleh view
			'tahunAjaran' => $tahunAjaranAktif, 
            'progressInputNilai' => $progressInputNilai,
            'chartData' => $chartData, // <--- VARIABEL KRITIS SUDAH DITAMBAHKAN
        ];
		    
        return view('guru/dashboard', $data);
    }
    
    /**
     * Menghitung progres input nilai, mengutamakan status final di nilai_rapor (100%).
     */
    protected function getProgressInputNilai(int $kelasId, int $tahunAjaranId, string $semester): array
    {
        // Asumsi: SiswaModel memiliki field 'kelas_id'
        $totalSiswa = $this->siswaModel->where('kelas_id', $kelasId)->countAllResults();

        if ($totalSiswa === 0) {
            return [];
        }

        // Asumsi: MataPelajaranModel memiliki field 'id' (mapel_id) dan 'nama_mapel'
        $mataPelajaranList = $this->mapelModel->findAll(); 
        $progress = [];

        foreach ($mataPelajaranList as $mapel) {
            $mapelId = $mapel['id']; 
            
            // A. Cek Finalisasi (Nilai Akhir di tabel 'nilai_rapor')
            $nilaiFinalCount = $this->nilaiRaporModel
                // Menggunakan kolom yang ada di NilaiRaporModel (kelas_id, semester, tahun_ajaran_id)
                ->where('kelas_id', $kelasId)
                ->where('tahun_ajaran_id', $tahunAjaranId)
                ->where('semester', $semester)
                ->where('mapel_id', $mapelId)
                ->countAllResults();

            if ($nilaiFinalCount === $totalSiswa && $totalSiswa > 0) {
                // KONDISI 1: SEMUA SUDAH FINAL DI NILAI AKHIR (100%)
                $persentase = 100; 
                $statusTeks = "Selesai";

            } else {
                // KONDISI 2: PROGRES DIHITUNG DARI INPUT KOMPONEN (tabel 'nilai')
                
                // --- START: LOGIKA DINAMIS MAX KOMPONEN ---
                
                // 1. Cari jumlah input terbanyak yang pernah dimasukkan oleh siswa manapun
                $mapelInputCounts = $this->nilaiModel
                    ->select('siswa_id, COUNT(id) as input_count')
                    ->where('kelas_id', $kelasId)
                    ->where('tahun_ajaran_id', $tahunAjaranId) 
                    ->where('semester', $semester) 
                    ->where('mapel_id', $mapelId)
                    ->groupBy('siswa_id')
                    ->orderBy('input_count', 'DESC')
                    ->first(); // Ambil yang paling tinggi (yang pertama setelah diurutkan DESC)

                $dynamicMaxKomponen = 0;
                if ($mapelInputCounts) {
                    $dynamicMaxKomponen = (int) $mapelInputCounts['input_count']; 
                }

                // Jika belum ada input, dynamicMaxKomponen akan 0, total expected 0, persentase 0. (BENAR)
                
                // --- END: LOGIKA DINAMIS MAX KOMPONEN ---
                
                
                $totalInputKomponen = $this->nilaiModel
                    ->where('kelas_id', $kelasId)
                    ->where('tahun_ajaran_id', $tahunAjaranId) 
                    ->where('semester', $semester) 
                    ->where('mapel_id', $mapelId)
                    ->countAllResults();

                $totalExpectedInput = $totalSiswa * $dynamicMaxKomponen;
                
                if ($totalExpectedInput > 0) {
                    $persentase = round(($totalInputKomponen / $totalExpectedInput) * 100);
                } else {
                    $persentase = 0;
                }
                
                // KOREKSI KRITIS: BATASI PERSENTASE MAKSIMAL DI 100%
                // untuk mencegah 125% dsb. jika ada input ekstra
                $persentase = min($persentase, 100); 
                
                // Jika sudah ada nilai final tersimpan (walau belum 100%), 
                if ($nilaiFinalCount > 0) {
                    $statusTeks = "Menunggu Finalisasi Siswa Lain";
                } else {
                    $statusTeks = "Proses Input Komponen";
                }
            }

            $progress[] = [
                'id' => $mapel['id'], // Tambahkan ID mapel untuk link di view
                'nama_mapel' => $mapel['nama_mapel'],
                'nilai_tersimpan_final_count' => $nilaiFinalCount,
                'total_siswa' => $totalSiswa,
                'persentase' => $persentase,
                'status_teks' => $statusTeks,
            ];
        }
        
        usort($progress, function($a, $b) {
            return $a['persentase'] <=> $b['persentase'];
        });

        return $progress;
    }
}