<?php

namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use App\Models\FisikAnakModel;
use App\Models\SiswaModel;
use App\Models\TahunAjaranModel;
use App\Models\KelasModel;

class FisikAnak extends BaseController
{
    protected $fisikAnakModel;
    protected $siswaModel;
    protected $tahunAjaranModel;
    protected $kelasModel;

    public function __construct()
    {
        $this->fisikAnakModel = new FisikAnakModel();
        $this->siswaModel = new SiswaModel();
        $this->tahunAjaranModel = new TahunAjaranModel();
        $this->kelasModel = new KelasModel();
    }

    // Fungsi helper untuk mendapatkan ID kelas yang diajar oleh guru yang sedang login (sebagai wali kelas)
    private function getGuruKelasIds()
    {
        $user_id = session()->get('user_id');
        $kelas_guru = $this->kelasModel->where('guru_kelas_id', $user_id)->findAll();
        
        $kelas_ids = [];
        foreach ($kelas_guru as $kelas) {
            $kelas_ids[] = $kelas['id_kelas'];
        }
        return $kelas_ids;
    }

    private function getActiveTahunAjaran()
    {
        return $this->tahunAjaranModel->where('aktif', 1)->first();
    }

    public function index()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (!$is_guru_kelas) {
            $data = [
                'title' => 'Data Fisik Anak',
                'fisik_anak' => [],
                'message' => 'Anda tidak terdaftar sebagai guru kelas. Halaman ini hanya bisa diakses oleh wali kelas.',
                'user'  => session()->get('user')
            ];
            return view('guru/fisik_anak/index', $data);
        }

        $tahun_ajaran = $this->getActiveTahunAjaran();
        
        if (empty($tahun_ajaran)) {
            $data = [
                'title' => 'Data Fisik Anak',
                'fisik_anak' => [],
                'message' => 'Tidak ada tahun ajaran atau semester aktif yang ditemukan.',
                'user'  => session()->get('user')
            ];
            return view('guru/fisik_anak/index', $data);
        }

        $fisikAnakList = $this->fisikAnakModel
                              ->select('fisik_anak.*, siswa.nama_lengkap, kelas.nama_kelas')
                              ->join('siswa', 'siswa.id = fisik_anak.siswa_id', 'left')
                              ->join('kelas', 'kelas.id_kelas = siswa.kelas_id', 'left')
                              ->whereIn('siswa.kelas_id', $guru_kelas_ids) // Filter berdasarkan kelas guru
                              ->where('fisik_anak.tahun_ajaran_id', $tahun_ajaran['id'])
                              ->where('fisik_anak.semester', $tahun_ajaran['semester'])
                              ->orderBy('siswa.nama_lengkap', 'ASC')
                              ->findAll();
        
        $data = [
            'title' => 'Data Fisik Anak',
            'fisik_anak' => $fisikAnakList,
            'tahun_ajaran_aktif' => $tahun_ajaran,
            'user'  => session()->get('user')
        ];
        return view('guru/fisik_anak/index', $data);
    }

    public function create()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (!$is_guru_kelas) {
            session()->setFlashdata('error', 'Halaman ini hanya bisa diakses oleh wali kelas.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }

        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (empty($tahun_ajaran)) {
            session()->setFlashdata('error', 'Tidak ada tahun ajaran atau semester aktif yang ditemukan.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }

        // Tampilkan hanya siswa dari kelas yang diajar oleh guru
        $siswa = $this->siswaModel
                      ->select('siswa.id, siswa.nama_lengkap, kelas.nama_kelas')
                      ->join('kelas', 'kelas.id_kelas = siswa.kelas_id')
                      ->whereIn('siswa.kelas_id', $guru_kelas_ids)
                      ->findAll();

        $data = [
            'title' => 'Tambah Data Fisik Anak',
            'siswa_list' => $siswa,
            'tahun_ajaran_aktif' => $tahun_ajaran,
            'validation' => service('validation'),
            'user'  => session()->get('user')
        ];
        return view('guru/fisik_anak/create', $data);
    }

    public function store()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (!$is_guru_kelas) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }
        
        $rules = [
            'siswa_id'      => 'required|numeric',
            'berat_badan'   => 'permit_empty|numeric',
            'tinggi_badan'  => 'permit_empty|numeric'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (empty($tahun_ajaran)) {
            session()->setFlashdata('error', 'Tidak ada tahun ajaran aktif yang ditemukan. Tidak dapat menyimpan.');
            return redirect()->back()->withInput();
        }
        
        $selected_siswa_id = $this->request->getVar('siswa_id');
        $siswa_info = $this->siswaModel->find($selected_siswa_id);

        if (!in_array($siswa_info['kelas_id'], $guru_kelas_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk menyimpan data fisik siswa di kelas tersebut.');
            return redirect()->back()->withInput();
        }

        $data = [
            'siswa_id'          => $selected_siswa_id,
            'berat_badan'       => $this->request->getVar('berat_badan'),
            'tinggi_badan'      => $this->request->getVar('tinggi_badan'),
            'semester'          => $tahun_ajaran['semester'],
            'tahun_ajaran_id'   => $tahun_ajaran['id'],
        ];

        $this->fisikAnakModel->save($data);

        session()->setFlashdata('success', 'Data fisik anak berhasil ditambahkan.');
        return redirect()->to(base_url('guru/fisik_anak'));
    }

    public function edit($id)
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (!$is_guru_kelas) {
            session()->setFlashdata('error', 'Halaman ini hanya bisa diakses oleh wali kelas.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }

        $fisikAnak = $this->fisikAnakModel->find($id);
        
        if (empty($fisikAnak)) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound();
        }
        
        $siswa = $this->siswaModel->find($fisikAnak['siswa_id']);
        
        if (!in_array($siswa['kelas_id'], $guru_kelas_ids)) {
             session()->setFlashdata('error', 'Anda tidak memiliki izin untuk mengedit data fisik anak di kelas tersebut.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }

        $data = [
            'title' => 'Edit Data Fisik Anak',
            'fisik_anak' => $fisikAnak,
            'siswa' => $siswa,
            'validation' => service('validation'),
            'user'  => session()->get('user')
        ];
        return view('guru/fisik_anak/edit', $data);
    }

    public function update($id)
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (!$is_guru_kelas) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }
        
        $rules = [
            'berat_badan'   => 'permit_empty|numeric',
            'tinggi_badan'  => 'permit_empty|numeric'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fisikAnak = $this->fisikAnakModel->find($id);
        $siswa = $this->siswaModel->find($fisikAnak['siswa_id']);
        
        if (!in_array($siswa['kelas_id'], $guru_kelas_ids)) {
             session()->setFlashdata('error', 'Anda tidak memiliki izin untuk memperbarui data fisik anak di kelas tersebut.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }

        $data = [
            'berat_badan'   => $this->request->getVar('berat_badan'),
            'tinggi_badan'  => $this->request->getVar('tinggi_badan'),
        ];

        $this->fisikAnakModel->update($id, $data);

        session()->setFlashdata('success', 'Data fisik anak berhasil diperbarui.');
        return redirect()->to(base_url('guru/fisik_anak'));
    }

    public function delete($id)
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (!$is_guru_kelas) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }
        
        $fisikAnak = $this->fisikAnakModel->find($id);
        if (empty($fisikAnak)) {
            session()->setFlashdata('error', 'Data fisik anak tidak ditemukan.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }

        $siswa = $this->siswaModel->find($fisikAnak['siswa_id']);
        
        if (!in_array($siswa['kelas_id'], $guru_kelas_ids)) {
             session()->setFlashdata('error', 'Anda tidak memiliki izin untuk menghapus data fisik anak di kelas tersebut.');
            return redirect()->to(base_url('guru/fisik_anak'));
        }
        
        $this->fisikAnakModel->delete($id);
        session()->setFlashdata('success', 'Data fisik anak berhasil dihapus.');
        return redirect()->to(base_url('guru/fisik_anak'));
    }
}