<?php

namespace App\Controllers\Admin;
use App\Controllers\BaseController;
use App\Models\UserModel; // Import Model User

class User extends BaseController
{
    protected $userModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
    }

    public function index()
    {
        $daftarUser = $this->userModel->findAll();

        $data = [
            'title'      => 'Manajemen Pengguna',
            'daftarUser' => $daftarUser
        ];
        return view('admin/users/index', $data);
    }

    public function create()
    {
        $data = [
            'title'      => 'Tambah Pengguna Baru',
            'validation' => \Config\Services::validation()
        ];
        return view('admin/users/create', $data);
    }

    public function store()
    {
        // Aturan validasi (password wajib diisi saat membuat user baru)
        if (!$this->validate([
            'username' => [
                'rules'  => 'required|min_length[4]|is_unique[users.username]',
                'errors' => ['required' => 'Username harus diisi.', 'min_length' => 'Username minimal 4 karakter.', 'is_unique' => 'Username ini sudah digunakan.']
            ],
            'password' => [
                'rules'  => 'required|min_length[6]',
                'errors' => ['required' => 'Password harus diisi.', 'min_length' => 'Password minimal 6 karakter.']
            ],
            'password_confirm' => [ // Untuk konfirmasi password
                'rules'  => 'required_with[password]|matches[password]',
                'errors' => ['required_with' => 'Konfirmasi password harus diisi.', 'matches' => 'Konfirmasi password tidak cocok dengan password.']
            ],
            'role'     => [
                'rules'  => 'required|in_list[admin,guru,kepala_sekolah]',
                'errors' => ['required' => 'Role harus dipilih.', 'in_list' => 'Role tidak valid.']
            ],
            'nama'     => [
                'rules'  => 'required|min_length[3]',
                'errors' => ['required' => 'Nama harus diisi.', 'min_length' => 'Nama minimal 3 karakter.']
            ],
            'nip'      => [
                'rules'  => 'permit_empty|min_length[5]|is_unique[users.nip]',
                'errors' => ['min_length' => 'NIP minimal 5 karakter.', 'is_unique' => 'NIP ini sudah digunakan.']
            ],
        ])) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        $this->userModel->save([
            'username' => $this->request->getVar('username'),
            'password' => $this->request->getVar('password'), // Password akan di-hash oleh callback di Model
            'role'     => $this->request->getVar('role'),
            'nama'     => $this->request->getVar('nama'),
            'nip'      => $this->request->getVar('nip'),
        ]);

        session()->setFlashdata('success', 'Pengguna berhasil ditambahkan.');
        return redirect()->to(base_url('admin/users'));
    }

    public function edit($id)
    {
        $user = $this->userModel->find($id);

        if (empty($user)) {
            session()->setFlashdata('error', 'Pengguna tidak ditemukan.');
            return redirect()->to(base_url('admin/user'));
        }

        $data = [
            'title'      => 'Edit Pengguna',
            'validation' => \Config\Services::validation(),
            'user'       => $user
        ];
        return view('admin/users/edit', $data);
    }

    public function update($id)
    {
        // Ambil data user yang ada untuk validasi is_unique
        $oldUser = $this->userModel->find($id);

        // Aturan validasi untuk update
        $rules = [
            'username' => 'required|min_length[4]',
            'role'     => 'required|in_list[admin,guru,kepala_sekolah]',
            'nama'     => 'required|min_length[3]',
            'nip'      => 'permit_empty|min_length[5]',
        ];

        // Jika username berubah, tambahkan aturan is_unique
        if ($oldUser['username'] !== $this->request->getVar('username')) {
            $rules['username'] .= '|is_unique[users.username,id,' . $id . ']';
        }
        // Jika NIP berubah, tambahkan aturan is_unique
        if ($oldUser['nip'] !== $this->request->getVar('nip')) {
            $rules['nip'] .= '|is_unique[users.nip,id,' . $id . ']';
        }

        // Jika password diisi (opsional saat update), tambahkan validasi password
        if ($this->request->getVar('password')) {
            $rules['password'] = 'required|min_length[6]';
            $rules['password_confirm'] = 'required_with[password]|matches[password]';
        }

        if (!$this->validate($rules, [
            'username' => [
                'required'    => 'Username harus diisi.',
                'min_length'  => 'Username minimal 4 karakter.',
                'is_unique'   => 'Username ini sudah digunakan.'
            ],
            'password' => [
                'required'   => 'Password harus diisi.',
                'min_length' => 'Password minimal 6 karakter.'
            ],
            'password_confirm' => [
                'required_with' => 'Konfirmasi password harus diisi.',
                'matches'       => 'Konfirmasi password tidak cocok dengan password.'
            ],
            'role' => [
                'required' => 'Role harus dipilih.',
                'in_list'  => 'Role tidak valid.'
            ],
            'nama' => [
                'required'   => 'Nama harus diisi.',
                'min_length' => 'Nama minimal 3 karakter.'
            ],
            'nip' => [
                'min_length' => 'NIP minimal 5 karakter.',
                'is_unique'  => 'NIP ini sudah digunakan.'
            ]
        ])) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        $dataToUpdate = [
            'username' => $this->request->getVar('username'),
            'role'     => $this->request->getVar('role'),
            'nama'     => $this->request->getVar('nama'),
            'nip'      => $this->request->getVar('nip'),
        ];

        // Hanya tambahkan password ke data update jika diisi di form
        if ($this->request->getVar('password')) {
            $dataToUpdate['password'] = $this->request->getVar('password'); // Akan di-hash oleh callback di Model
        }

        $this->userModel->update($id, $dataToUpdate);

        session()->setFlashdata('success', 'Data pengguna berhasil diperbarui.');
        return redirect()->to(base_url('admin/users'));
    }

    public function delete($id)
    {
        $this->userModel->delete($id);
        session()->setFlashdata('success', 'Pengguna berhasil dihapus.');
        return redirect()->to(base_url('admin/users'));
    }
}